/**
 * Text processing utilities for finding and replacing citation patterns
 */

/**
 * Process text node for citation patterns
 */
window.TextProcessor = class {
  constructor(patterns, linkStyle, showTooltips) {
    this.patterns = patterns || [];
    this.linkStyle = linkStyle || 'dashed';
    this.showTooltips = showTooltips || true;
    this.compiledPatterns = this.compilePatterns();
  }
  
  /**
   * Compile regex patterns for performance
   */
  compilePatterns() {
    return this.patterns.map(pattern => {
      try {
        return {
          ...pattern,
          compiledRegex: new RegExp(pattern.regex, 'gi')
        };
      } catch (error) {
        console.warn(`Failed to compile pattern ${pattern.name}:`, error);
        return null;
      }
    }).filter(Boolean);
  }
  
  /**
   * Process a text node and return matches
   */
  findMatches(textContent) {
    const matches = [];
    
    for (const pattern of this.compiledPatterns) {
      pattern.compiledRegex.lastIndex = 0; // Reset regex state
      let match;
      
      while ((match = pattern.compiledRegex.exec(textContent)) !== null) {
        const url = window.generateUrl(pattern.baseUrl, match.slice(1));
        
        if (url) {
          matches.push({
            pattern: pattern,
            match: match[0],
            groups: match.slice(1),
            index: match.index,
            url: url,
            endIndex: match.index + match[0].length
          });
        }
        
        // Prevent infinite loop on zero-length matches
        if (match.index === pattern.compiledRegex.lastIndex) {
          pattern.compiledRegex.lastIndex++;
        }
      }
    }
    
    // Sort matches by index and remove overlaps (first match wins)
    matches.sort((a, b) => a.index - b.index);
    return this.removeOverlaps(matches);
  }
  
  /**
   * Remove overlapping matches, keeping the first one
   */
  removeOverlaps(matches) {
    const filtered = [];
    
    for (const match of matches) {
      const hasOverlap = filtered.some(existing => 
        (match.index >= existing.index && match.index < existing.endIndex) ||
        (existing.index >= match.index && existing.index < match.endIndex)
      );
      
      if (!hasOverlap) {
        filtered.push(match);
      }
    }
    
    return filtered;
  }
  
  /**
   * Create a citation link element
   */
  createCitationLink(match) {
    const link = document.createElement('a');
    link.href = match.url;
    link.target = '_blank';
    link.rel = 'noopener noreferrer';
    link.textContent = match.match;
    link.className = `cite-linker-pro-link cite-linker-pro-${this.linkStyle}`;
    
    // Add data attributes for identification
    link.setAttribute('data-cite-linker-pro', 'true');
    link.setAttribute('data-pattern-id', match.pattern.id);
    link.setAttribute('data-original-text', match.match);
    
    // Add tooltip if enabled
    if (this.showTooltips) {
      link.title = `${match.pattern.name}: ${match.url}`;
      link.setAttribute('data-cite-tooltip', match.url);
    }
    
    return link;
  }
  
  /**
   * Process a text node and replace matches with links
   */
  processTextNode(textNode) {
    if (!textNode || textNode.nodeType !== Node.TEXT_NODE) {
      return false;
    }
    
    const parent = textNode.parentNode;
    if (!parent) {
      return false;
    }
    
    // Skip if already inside a link or cite-linker element
    if (this.isInsideLinkOrCitation(textNode)) {
      return false;
    }
    
    const textContent = textNode.textContent;
    const matches = this.findMatches(textContent);
    
    if (matches.length === 0) {
      return false;
    }
    
    // Create document fragment with replaced content
    const fragment = document.createDocumentFragment();
    let lastIndex = 0;
    
    for (const match of matches) {
      // Add text before match
      if (match.index > lastIndex) {
        const beforeText = textContent.slice(lastIndex, match.index);
        fragment.appendChild(document.createTextNode(beforeText));
      }
      
      // Add citation link
      const link = this.createCitationLink(match);
      fragment.appendChild(link);
      
      lastIndex = match.endIndex;
    }
    
    // Add remaining text
    if (lastIndex < textContent.length) {
      const afterText = textContent.slice(lastIndex);
      fragment.appendChild(document.createTextNode(afterText));
    }
    
    // Replace the text node with the fragment
    parent.replaceChild(fragment, textNode);
    return true;
  }
  
  /**
   * Check if text node is inside a link or existing citation
   */
  isInsideLinkOrCitation(node) {
    let current = node.parentElement;
    
    while (current) {
      const tagName = current.tagName?.toLowerCase();
      
      if (tagName === 'a' || 
          tagName === 'script' || 
          tagName === 'style' || 
          tagName === 'noscript' ||
          current.hasAttribute('data-cite-linker-pro')) {
        return true;
      }
      
      current = current.parentElement;
    }
    
    return false;
  }
  
  /**
   * Update patterns and recompile
   */
  updatePatterns(patterns) {
    this.patterns = patterns;
    this.compiledPatterns = this.compilePatterns();
  }
  
  /**
   * Update link style
   */
  updateLinkStyle(linkStyle) {
    this.linkStyle = linkStyle;
  }
  
  /**
   * Update tooltip setting
   */
  updateTooltips(showTooltips) {
    this.showTooltips = showTooltips;
  }
}