/**
 * Predefined citation patterns loader - loads from bundled XML file
 */

// Fallback patterns if XML loading fails
window.FALLBACK_PREDEFINED_PATTERNS = [
  {
    id: 'us-supreme-court',
    name: 'U.S. Supreme Court Citations',
    regex: '(\\d+)\\s+U\\.?S\\.?\\s+(\\d+)',
    baseUrl: 'https://supreme.justia.com/cases/federal/us/{0}/{1}/',
    description: 'Matches citations like "410 U.S. 113" or "555 US 123"',
    enabled: true
  }
];

// Cached predefined patterns
let cachedPredefinedPatterns = null;
let patternsLoadPromise = null;

/**
 * Load predefined patterns from bundled XML file
 */
async function loadPredefinedPatternsFromXML() {
  try {
    const xmlUrl = chrome.runtime.getURL('predefined-patterns.xml');
    const response = await fetch(xmlUrl);
    
    if (!response.ok) {
      throw new Error(`Failed to load predefined patterns: ${response.status}`);
    }
    
    const xmlText = await response.text();
    const importResult = window.importFromXMLAny ? window.importFromXMLAny(xmlText) : null;
    
    if (importResult && importResult.success && importResult.data.patterns) {
      console.log(`Loaded ${importResult.data.patterns.length} predefined patterns from XML`);
      return importResult.data.patterns.map(pattern => ({
        ...pattern,
        isPredefined: true
      }));
    } else {
      throw new Error('Failed to parse predefined patterns XML');
    }
  } catch (error) {
    console.warn('Failed to load predefined patterns from XML:', error);
    console.log('Using fallback predefined patterns');
    return window.FALLBACK_PREDEFINED_PATTERNS.map(pattern => ({
      ...pattern,
      isPredefined: true
    }));
  }
}

/**
 * Get all predefined patterns (async)
 */
window.getPredefinedPatterns = async function() {
  if (cachedPredefinedPatterns) {
    return cachedPredefinedPatterns;
  }
  
  if (!patternsLoadPromise) {
    patternsLoadPromise = loadPredefinedPatternsFromXML();
  }
  
  cachedPredefinedPatterns = await patternsLoadPromise;
  return cachedPredefinedPatterns;
}

/**
 * Get all predefined patterns synchronously (for compatibility)
 * Returns cached patterns or fallback if not loaded yet
 */
window.getPredefinedPatternsSync = function() {
  if (cachedPredefinedPatterns) {
    return cachedPredefinedPatterns;
  }
  
  // Return fallback patterns if not loaded yet
  return window.FALLBACK_PREDEFINED_PATTERNS.map(pattern => ({
    ...pattern,
    isPredefined: true
  }));
}

/**
 * Get enabled predefined patterns
 */
window.getEnabledPredefinedPatterns = async function() {
  const patterns = await window.getPredefinedPatterns();
  return patterns.filter(pattern => pattern.enabled !== false);
}

/**
 * Initialize predefined patterns loading
 */
window.initializePredefinedPatterns = function() {
  // Start loading patterns immediately
  if (!patternsLoadPromise) {
    patternsLoadPromise = loadPredefinedPatternsFromXML();
  }
  return patternsLoadPromise;
}

/**
 * Clear cached predefined patterns (force reload)
 */
window.clearPredefinedPatternsCache = function() {
  cachedPredefinedPatterns = null;
  patternsLoadPromise = null;
}